import os
import time
import logging
import requests
import sqlite3
from datetime import datetime, timezone, timedelta
from dateutil import tz, parser
from flask import Flask, render_template, jsonify, request, redirect, url_for
from dotenv import load_dotenv
load_dotenv()

# 🔐 Load environment variables
load_dotenv()
API_KEY = os.getenv("YOUTUBE_API_KEY")
LIVE_CHAT_ID = os.getenv("LIVE_CHAT_ID")
FILTER_KEYWORD = os.getenv("FILTER_KEYWORD", "Ææ")

# 🛠️ Flask app
app = Flask(__name__)
LOCAL_ZONE = tz.gettz("Europe/Stockholm")
last_fetch_time = 0

# 📋 Logging
logging.basicConfig(
    filename="error.log",
    level=logging.INFO,
    format="%(asctime)s: %(message)s",
    filemode="a"
)

# 🗂️ Database setup
def get_db_connection():
    return sqlite3.connect("chat_archive.db")

def init_db():
    conn = get_db_connection()
    cursor = conn.cursor()
    cursor.execute("""CREATE TABLE IF NOT EXISTS live_chat (
        id TEXT PRIMARY KEY,
        author TEXT,
        message TEXT,
        timestamp TEXT
    )""")
    cursor.execute("""CREATE TABLE IF NOT EXISTS visitors (
        ip TEXT PRIMARY KEY,
        visit_time TEXT
    )""")
    conn.commit()
    conn.close()

init_db()

# 👥 Visitor counter
def count_unique_visitors():
    ip = request.headers.get("X-Forwarded-For", request.remote_addr)
    visit_time = datetime.utcnow().strftime("%Y-%m-%d %H:%M:%S")
    conn = get_db_connection()
    cursor = conn.cursor()
    cursor.execute("INSERT OR IGNORE INTO visitors VALUES (?, ?)", (ip, visit_time))
    conn.commit()
    cursor.execute("SELECT COUNT(*) FROM visitors")
    count = cursor.fetchone()[0]
    conn.close()
    return count

# 🛰️ Fetch live chat
def fetch_live_chat():
    if not API_KEY or not LIVE_CHAT_ID:
        logging.warning("Missing API_KEY or LIVE_CHAT_ID.")
        return []

    url = "https://www.googleapis.com/youtube/v3/liveChat/messages"
    params = {
        "liveChatId": LIVE_CHAT_ID,
        "part": "snippet,authorDetails",
        "key": API_KEY,
        "maxResults": 200
    }

    try:
        response = requests.get(url, params=params)
        data = response.json()
        comments = []

        conn = get_db_connection()
        cursor = conn.cursor()

        for item in data.get("items", []):
            chat_id = item["id"]
            author = item["authorDetails"]["displayName"]
            message = item["snippet"]["displayMessage"]
            raw_time = item["snippet"]["publishedAt"]

            utc_time = parser.parse(raw_time).replace(tzinfo=timezone.utc)
            local_time = utc_time.astimezone(LOCAL_ZONE).strftime("%Y-%m-%d %H:%M:%S")

            if FILTER_KEYWORD and FILTER_KEYWORD.lower() not in message.lower():
                continue

            comments.append({
                "author": author,
                "message": message,
                "timestamp": local_time
            })
            cursor.execute("INSERT OR IGNORE INTO live_chat VALUES (?, ?, ?, ?)",
                           (chat_id, author, message, local_time))

        conn.commit()
        conn.close()
        return comments

    except Exception as e:
        logging.error(f"Fetch error: {e}")
        return []

# 🔁 Auto-fetch before request
@app.before_request
def maybe_fetch_chat():
    global last_fetch_time
    now = time.time()
    force = request.args.get("refresh") == "now"
    if force or (now - last_fetch_time > 30):
        fetched = fetch_live_chat()
        last_fetch_time = now
        logging.info(f"Fetched {len(fetched)} comments (forced={force})")

# 🏠 Home route
@app.route("/home")
def home():
    visitors = count_unique_visitors()
    return render_template("index.html",
                           visitor_count=visitors,
                           current_year=datetime.utcnow().year)

@app.route("/")
def landing():
    return redirect(url_for('home'))

# 📡 Live comments API
@app.route("/comments")
def comments():
    conn = get_db_connection()
    cursor = conn.cursor()
    cursor.execute("""
        SELECT author, message, timestamp
        FROM live_chat
        WHERE date(timestamp) = date('now', 'localtime')
        ORDER BY timestamp DESC
    """)
    rows = cursor.fetchall()
    conn.close()
    return jsonify([
        {"author": r[0], "message": r[1], "timestamp": r[2]} for r in rows
    ])

# 📚 Archive by month
@app.route("/archive/by_month")
def archive_by_month():
    conn = get_db_connection()
    cursor = conn.cursor()
    cursor.execute("""
        SELECT DISTINCT strftime('%Y-%m', timestamp) AS month
        FROM live_chat
        ORDER BY month DESC
    """)
    months = [row[0] for row in cursor.fetchall()]

    selected_month = request.args.get("month")
    if selected_month:
        cursor.execute("""
            SELECT author, message, timestamp
            FROM live_chat
            WHERE strftime('%Y-%m', timestamp) = ?
            ORDER BY timestamp DESC
        """, (selected_month,))
        comments = cursor.fetchall()
    else:
        comments = []

    conn.close()
    return render_template("archive.html",
                           comments=comments,
                           months=months,
                           selected_month=selected_month)

# 🧮 Week helpers
def get_week_range(year, week_number):
    monday = datetime.strptime(f'{year}-W{week_number}-1', "%Y-W%W-%w")
    sunday = monday + timedelta(days=6)
    return monday.strftime("%b %d"), sunday.strftime("%b %d")

# 📚 Archive by week
@app.route("/archive/by_week")
def archive_by_week():
    conn = get_db_connection()
    cursor = conn.cursor()
    cursor.execute("""
        SELECT DISTINCT strftime('%Y-%m', timestamp) AS month
        FROM live_chat
        ORDER BY month DESC
    """)
    months = [row[0] for row in cursor.fetchall()]

    selected_week = request.args.get("week")
    selected_month = request.args.get("month")
    selected_label = None

    if selected_week:
        cursor.execute("""
            SELECT author, message, timestamp
            FROM live_chat
            WHERE strftime('%Y-%W', timestamp) = ?
            ORDER BY timestamp DESC
        """, (selected_week,))
        comments = cursor.fetchall()

        year, week = selected_week.split("-")
        start, end = get_week_range(int(year), int(week))
        selected_label = f"Week {week} ({start}–{end})"
    else:
        comments = []

    conn.close()
    return render_template("archive.html",
                           comments=comments,
                           months=months,
                           selected_month=selected_month,
                           selected_week=selected_week,
                           selected_label=selected_label)

# 📅 Get weeks in a month
@app.route("/get_weeks", methods=["POST"])
def get_weeks():
    selected_month = request.form.get("month")
    conn = get_db_connection()
    cursor = conn.cursor()
    cursor.execute("""
        SELECT DISTINCT strftime('%Y-%W', timestamp) AS week
        FROM live_chat
        WHERE strftime('%Y-%m', timestamp) = ?
        ORDER BY week ASC
    """, (selected_month,))
    raw_weeks = [row[0] for row in cursor.fetchall()]

    weeks = []
    for w in raw_weeks:
        year, week = w.split("-")
        start, end = get_week_range(int(year), int(week))
        label = f"Week {week} ({start}–{end})"
        weeks.append({"value": w, "label": label})

    conn.close()
    return jsonify(weeks)

# 🔄 Reset database
@app.route("/reset")
def reset():
    conn = get_db_connection()
    cursor = conn.cursor()
    cursor.execute("DELETE FROM live_chat")
    cursor.execute("DELETE FROM visitors")
    conn.commit()
    conn.close()
    return "Database reset! 🔄"

# 🩺 Ping
@app.route("/ping")
def ping():
    return "pong"

# 🚀 Run app
if __name__ == "__main__":
    app.run(host="0.0.0.0", port=5000, debug=True)
